<?php

namespace TenWebWooP\PaymentMethods;

use Automattic\WooCommerce\StoreApi\Exceptions\RouteException;
use TenWebWooP\Config;

class PaymentMethod extends \WC_Payment_Gateway {

    protected static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    public static function register_payment_method() {
        add_filter('woocommerce_payment_gateways', array(self::class, 'add_gateway_class'));

        add_action('admin_enqueue_scripts', array( self::get_instance(), 'enqueue_admin_scripts' ));
        add_action('wp_enqueue_scripts', array( self::get_instance(), 'enqueue_scripts' ));
    }

    public static function add_gateway_class($methods) {
        $methods[] = self::class;

        return $methods;
    }

    public function __construct() {
        $this->id = 'tenweb_payengine';
        $this->icon = '';
        $this->has_fields = true;
        $this->title = 'Tenweb Payments';
        $this->method_title = 'Tenweb Payments';
        $this->supports = array(
            'products',
            'refunds',
        );
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ));

        $this->init_form_fields();
        $this->init_settings();
        $this->mode = $this->get_option('test_mode') === 'no' ? 'live' : 'test';

        $this->merchant = Config::get_payengine_data($this->mode);
        $this->method_description = $this->get_method_description();
    }

    public function process_admin_options() {
        parent::process_admin_options();
    }

    public function enqueue_admin_scripts() {
        wp_enqueue_script('twwp_admin_script', Config::get_url('PaymentMethods', '/assets/admin.js'), array('jquery'), Config::VERSION);
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'twwp'),
                'type' => 'checkbox',
                'label' => __($this->method_title, 'twwp'),
                'default' => 'yes'
            ),
            'title' => array(
                'title' => __('Title', 'twwp'),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'twwp'),
                'default' => __($this->method_title, 'twwp'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Customer Message', 'twwp'),
                'type' => 'textarea',
                'default' => ''
            ),
            'test_mode' => array(
                'title' => __('Test mode', 'twwp'),
                'label' => __('Enable test mode', 'twwp'),
                'type' => 'checkbox',
                'description' => __('Simulate transactions using test card numbers.', 'twwp'),
                'default' => 'yes',
                'desc_tip' => true,
            ),
        );
    }

    /**
     * Get the right method description if WooPay is eligible.
     *
     * @return string
     */
    public function get_method_description() {
        return 'Tenweb Payment Description<br />Currnet merchant is: ' . $this->merchant['merchant_id'];
    }

    public function enqueue_scripts() {
        wp_enqueue_script('twwp_payengine', $this->merchant['script_url'], array( 'jquery' ), null, false);
        wp_enqueue_script('twwp_script', Config::get_url('PaymentMethods', 'assets/script.js'), array('jquery'), Config::VERSION);
        wp_localize_script(
            'twwp_script',
            'twwp_config',
            array(
                'merchant_id' => $this->merchant['merchant_id'],
            )
        );
    }

    public function payment_fields() {
        // TODO: Show a notif if test mode is enabled.
        // Replacing the form with a div in Elementor preview as it is removed in preview mode.
        $tagname = \Elementor\Plugin::instance()->preview->is_preview_mode() || is_preview() ? 'form' : 'div'; ?>
      <<?php echo esc_attr($tagname); ?> id="twwp-card-form" onsubmit="return false">
        <style>
          div.form-field{height: 50px;}
          #twwp-card-form {
            display: flex;
            flex-wrap: wrap;
            gap: 0 10px;
          }
          #twwp-card-form .form-field#cc-name,
          #twwp-card-form .form-field#cc-number {
            flex: 1 0 100%;
          }
          #twwp-card-form .form-field#cc-expiration-date,
          #twwp-card-form .form-field#cc-cvc {
            flex: 0 0 calc(50% - 5px);
          }
          .form-field iframe {
            max-width: 100%;
            max-height: 100%;
          }
        </style>

        <div class="form-field" id="cc-name"></div>
        <div class="form-field" id="cc-number"></div>
        <div class="form-field" id="cc-expiration-date"></div>
        <div class="form-field" id="cc-cvc"></div>
      </<?php echo esc_attr($tagname); ?>>
      <?php
    }

    public function process_payment($order_id) {
        // TODO: Display a notif for orders which were payed in test mode based on 'twwp_environment' meta
        global $woocommerce;
        $order = new \WC_Order($order_id);

        if ($order->get_total() > 0) {
            $merchant_id = $this->merchant['merchant_id'];
            //phpcs:ignore WordPress.Security.NonceVerification.Missing
            $card_token = isset($_POST['twwp_payengine_card_token']) ? sanitize_text_field($_POST['twwp_payengine_card_token']) : '';

            // TODO: add all available data here
            $args = array(
                'data' => array(
                    'transactionAmount' => number_format($order->get_total(), 2, '.', ''),
                    'cardToken' => $card_token,
                    'currencyCode' => $order->get_currency(),
                    'internalTransactionID' => $order_id,
                    // phpcs:ignore
                    // 'items' => array(),
                ),
            );
            //phpcs:ignore
            /*foreach ($order->get_items() as $item) {
              $args['data']['items'][] = array(
                'name' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'totalAmount' => $item->get_(),
              );
            }*/
            $payment = Service::request('merchants/' . $merchant_id . '/' . $this->mode . '/payment/sale', $args, 'POST');

            if ($payment && isset($payment->data->SaleResponse->status) && 'PASS' === $payment->data->paymentStatus) {
                $order->set_transaction_id($payment->data->TransactionID);
                $order->update_meta_data('twwp_merchant_id', $payment->data->MerchantID);
                $order->update_meta_data('twwp_transaction', $payment->data);
                $order->update_meta_data('twwp_environment', $payment->data->payengineEnv);
                $order->update_status('completed');
                $order->add_order_note('Payment succeeded. Merchant ID: ' . $payment->data->MerchantID . '. Transaction ID: ' . $payment->data->TransactionID . '.');

                $woocommerce->cart->empty_cart();

                return array(
                    'result' => 'success',
                    'redirect' => $this->get_return_url($order)
                );
            } else {
                $order->add_order_note('Payment failed. Merchant ID: ' . $payment->data->MerchantID . '. Transaction ID: ' . $payment->data->TransactionID . '. Error Code: ' . $payment->data->SaleResponse->responseCode . '. Error Message: ' . $payment->data->SaleResponse->responseMessage . '.');
                throw new RouteException('woocommerce_rest_checkout_process_payment_error', __('Payment failed', 'twwp'), 402);
            }
        } else {
            $order->payment_complete();
            // Remove cart.
            WC()->cart->empty_cart();

            // Return thankyou redirect.
            return array(
                'result' => 'success',
                'redirect' => $this->get_return_url($order),
            );
        }
    }

    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = new \WC_Order($order_id);
        $merchant_id = $this->merchant['merchant_id'];
        $transaction = $order->get_meta('twwp_transaction');
        $environment = $order->get_meta('twwp_environment');
        $args = array(
            'data' => array(
                'transactionAmount' => number_format($amount, 2, '.', ''),
                'currencyCode' => $order->get_currency(),
                'internalTransactionID' => $order_id,
            ),
        );
        $payment = Service::request('merchants/' . $merchant_id . '/' . $environment . '/payment/sale', $args, 'POST');

        return true;
    }
}
