<?php

namespace TenWebWooP;

use TenWebWooP\PaymentMethods\OrderActions;
use TenWebWooP\WoocommerceData\WoocommerceDataRepository;
use WP_REST_Request;
use WP_REST_Server;

class Api {

    const SETTING_OPTIONS_FOR_PRODUCT_CREATION = array(
        'woocommerce_weight_unit'
    );

    use CheckAuthorization;
    public $media_handler;

    public $payments_handler;

    private $request_from_proxy = true;

    private $data_repository = null;

    private $jwt_token = array();

    public function __construct() {
        $this->data_repository = new WoocommerceDataRepository();
        $this->media_handler = new Media('attachment');
        $this->payments_handler = new PaymentMethods\Api();
        add_action('rest_api_init', array($this, 'tenweb_woocommerce_rest'));
    }

    public function tenweb_woocommerce_rest() {
        register_rest_route(
            'tenweb_woop/v1',
            'products',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'products'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->products->get_collection_params(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'orders',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'orders'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->orders->get_collection_params(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'orders/(?P<id>[\d]+)',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'order'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->orders->get_collection_params(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'customers',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'customers'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->customers->get_collection_params(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'customers/(?P<id>[\d]+)',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'customer'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->customers->get_collection_params(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'reports/customers',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'customers_reports'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->customersReports->get_collection_params(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'jwt_token',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'get_jwt_token'),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'shop_info',
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_shop_info'),
                    'permission_callback' => array($this, 'check_rest_auth'),
                ),
                array(
                    'methods' => WP_REST_Server::CREATABLE,
                    'callback' => array($this, 'set_shop_info'),
                    'permission_callback' => array($this, 'check_rest_auth'),
                )
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'product_with_variations',
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'product_with_variations'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->products->get_endpoint_args_for_item_schema(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'product_with_variations/(?P<id>[\d]+)',
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'product_with_variations'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->products->get_endpoint_args_for_item_schema(),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'edit_product',
            array(
                'methods' => 'GET',
                'callback' => array($this, 'edit_product'),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'set_up_info',
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'set_up_info'),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );

        register_rest_route(
            'tenweb_woop/v1',
            'settings',
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_settings'),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'settings',
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'set_settings'),
                'permission_callback' => array($this, 'check_rest_auth'),
                'args' => $this->data_repository->settingsOptions->get_endpoint_args_for_item_schema(WP_REST_Server::EDITABLE),
            )
        );

        register_rest_route(
            'tenweb_woop/v1',
            'set_default_country',
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'set_default_country'),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );

        register_rest_route(
            'tenweb_woop/v1',
            'payment_test_mode',
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'set_payment_test_mode'),
                'args' => array(
                    'mode' => array(
                        'type' => 'string',
                        'required' => true,
                        'enum' => array('no', 'yes')
                    ),
                ),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );
        register_rest_route(
            'tenweb_woop/v1',
            'get_emails',
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_emails'),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );

        register_rest_route(
            'tenweb_woop/v1',
            'payment_methods_positions',
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'set_payment_methods_positions'),
                'args' => array(
                    'gateways' => array(
                        'type' => 'string',
                        'required' => true,
                        'validate_callback' => array($this, 'validate_unique_positions'),
                    ),
                ),
                'permission_callback' => array($this, 'check_rest_auth'),
            )
        );

        $this->media_handler->register_routes();
        $this->payments_handler->register_routes();
    }

    public function set_payment_methods_positions(WP_REST_Request $request) {
        $payment_gateways = WC()->payment_gateways->payment_gateways();
        $gateways_data = $request->get_param('gateways');
        $gateways_data = json_decode($gateways_data, true);
        $order = get_option('woocommerce_gateway_order', array());

        $new_order = $order;

        foreach ($gateways_data as $gateway_data) {
            if (isset($gateway_data['id'], $gateway_data['position'])) {
                $gateway_id = $gateway_data['id'];

                if (isset($payment_gateways[$gateway_id])) {
                    $gateway = $payment_gateways[$gateway_id];

                    if (isset($gateway_data['enabled'])) {
                        $settings = $gateway->settings;
                        $settings['enabled'] = wc_bool_to_string($gateway_data['enabled']);
                        $gateway->enabled = $settings['enabled'];
                        $gateway->settings = $settings;
                        update_option($gateway->get_option_key(), apply_filters('woocommerce_gateway_' . $gateway->id . '_settings_values', $settings, $gateway));
                    }
                }

                $new_order[ $gateway_id ] = (int) $gateway_data['position'];
            }
        }

        update_option('woocommerce_gateway_order', $new_order);

        return $this->modify_response(array('success' => true));
    }

    public function get_emails() {
        $wc_emails = $this->data_repository->getEmails();

        if (!empty($wc_emails)) {
            return $this->modify_response($wc_emails);
        }

        return $this->modify_response(array());
    }

    /**
     * @return array|mixed
     */
    public function get_settings(WP_REST_Request $request) {
        $return_data = array();
        $settings_data = $this->data_repository->getSettings();
        $return_data['settings'] = $settings_data['settings'];
        $return_data['country_currency_codes'] = $settings_data['country_currency_codes'];

        return $this->modify_response($return_data);
    }

    /**
     * @return string[]
     */
    public function set_settings(WP_REST_Request $request) {
        $data = $this->data_repository->setSettings($request);

        return $this->modify_response($data);
    }

    /**
     * This function for get all customers from reports.
     * This endpoint is used for populating all the customers including the ones without registration
     * Limit result set to resources with a specific role.
     * Options: all, administrator, editor, author, contributor, subscriber, customer, shop_manager and guest. Default is customer
     *
     * @param WP_REST_Request $request full details about the request
     *
     * @return array {
     *
     * @var int   $customersTotals all customers count
     * @var array $customers       Registered customers list.
     *            }
     */
    public function customers_reports(WP_REST_Request $request) {
        $return_data = array();
        $customersReportTotals = $this->data_repository->customersReportTotals();
        $customersReport = $this->data_repository->customersReport($request, true);
        $return_data['customersTotals'] = $customersReportTotals;
        $return_data['customers'] = $customersReport;

        return $this->modify_response($return_data);
    }

    /**
     * @return false|mixed
     */
    public function check_rest_auth(WP_REST_Request $request) {
        $headers = Utils::getAllHeaders();

        if (!isset($headers['X-TENWEB-SERVICE-REQUEST'])) {
            $this->request_from_proxy = false;
        }
        $check_authorization = $this->check_authorization($request);

        if (is_array($check_authorization)) {
            if (!empty($check_authorization['jwt_token'])) {
                $this->jwt_token = $check_authorization['jwt_token'];
            }

            return $check_authorization['success'];
        }

        return false;
    }

    /**
     * This function for get jwt token.
     *
     * @return array jwt token
     */
    public function get_jwt_token() {
        return $this->jwt_token;
    }

    /**
     * @return array
     */
    public function product_with_variations(WP_REST_Request $request) {
        $return_data = array();

        if (isset($request['id'])) {
            $product = $this->data_repository->products->update_item($request);
        } else {
            $product = $this->data_repository->products->create_item($request);
        }

        $return_data['product'] = $product;
        $args = $request->get_url_params();
        $args['product_id'] = $product->data['id'];
        $request->set_url_params($args);
        $variations = $this->data_repository->productsVariations->batch_items($request);
        $return_data['variations'] = $variations;

        $request->set_param('per_page', 25);
        $request->set_param('order', 'desc');
        $products = $this->data_repository->products($request);
        $return_data['products'] = $products->data;

        return $return_data;
    }

    /**
     * @return array|mixed
     */
    public function edit_product(WP_REST_Request $request) {
        $request_custom_params = array(
            'per_page' => 10000
        );
        $return_data = array();
        $categories = $this->data_repository->categories($request, true, $request_custom_params);
        $tags = $this->data_repository->tags($request, true, $request_custom_params);
        $shipping_class_list = $this->data_repository->shippingClasses($request, true);

        $products_request = $request;
        $products_request->set_param('per_page', 25);
        $products_request->set_param('order', 'desc');
        $products = $this->data_repository->products($products_request);
        $products_attributes = $this->data_repository->productsAttributes($products_request, true);

        $settings_data = $this->data_repository->settingsOptions('products', $request, true);
        $return_data['settings'] = array(
            'products' => array_column(array_filter(
                $settings_data->data,
                function ($setting) {
                    return in_array($setting['id'], self::SETTING_OPTIONS_FOR_PRODUCT_CREATION, true);
                }
            ), null, 'id')
        );
        $return_data['categories'] = $categories->data;
        $return_data['tags'] = $tags->data;
        $return_data['shipping_class_list'] = $shipping_class_list->data;
        $return_data['products'] = $products->data;
        $return_data['products_attributes'] = $products_attributes->data;

        return $this->modify_response($return_data);
    }

    /**
     * @return false|mixed|null
     */
    public function get_shop_info() {
        return Settings::get('shop_info');
    }

    /**
     * @return string[]
     */
    public function set_shop_info(WP_REST_Request $request) {
        $shop_info = $request->get_param('shop_info');
        Settings::update('shop_info', $shop_info);

        return array('Success');
    }

    /**
     * This function for get products data.
     *
     * @param WP_REST_Request $request full details about the request
     *
     * @return array {
     *
     * @var int   $productsTotals all products count
     * @var array $products       products list per page
     * @var array $categories     all products categories
     * @var array $tags           All products categories.
     *            }
     */
    public function products(WP_REST_Request $request) {
        $status = $request->get_param('status');
        $return_data = array();
        $productsTotals = $this->data_repository->productsTotals($status);

        $products = $this->data_repository->products($request);
        $request_custom_params = array(
            'per_page' => 10000
        );
        $categories = $this->data_repository->categories($request, true, $request_custom_params);
        $tags = $this->data_repository->tags($request, true, $request_custom_params);
        $return_data['productsTotals'] = $productsTotals;
        $return_data['products'] = $products->data;
        $return_data['categories'] = $categories->data;
        $return_data['tags'] = $tags->data;

        return $this->modify_response($return_data);
    }

    /**
     * This function for get orders data.
     *
     * @param WP_REST_Request $request full details about the request
     *
     * @return array {
     *
     * @var int   $ordersTotals all orders count
     * @var array $orders       Products list per page.
     *            }
     */
    public function orders(WP_REST_Request $request) {
        $return_data = array();
        $ordersTotals = $this->data_repository->ordersTotals();
        $orders = $this->data_repository->orders($request);
        $return_data['ordersTotals'] = $ordersTotals;
        $return_data['orders'] = $orders;
        $return_data['test_mode'] = Utils::getTestMode();

        return $this->modify_response($return_data);
    }

    /**
     * This function for get single order.
     *
     * @param WP_REST_Request $request full details about the request
     *
     * @return array Single order data
     */
    public function order(WP_REST_Request $request) {
        $return_data = $this->data_repository->order($request);

        return $this->modify_response($return_data);
    }

    /**
     * This function for get customers if customer registered user.
     * Limit result set to resources with a specific role.
     * Options: all, administrator, editor, author, contributor, subscriber, customer and shop_manager. Default is customer
     *
     * @param WP_REST_Request $request full details about the request
     *
     * @depecated please use Api::customers_reports() function
     *
     * @return array {
     *
     * @var int   $customersTotals all orders count
     * @var array $customers       Registered customers list.
     *            }
     */
    public function customers(WP_REST_Request $request) {
        $return_data = array();
        $customersTotals = $this->data_repository->customersTotals();
        $return_data['customersTotals'] = $customersTotals;
        $return_data['customers'] = $this->data_repository->customers($request);

        return $this->modify_response($return_data);
    }

    /**
     * This function for get customer by customer_id.
     *
     * @param WP_REST_Request $request full details about the request
     *
     * @return array Customer data
     */
    public function customer(WP_REST_Request $request) {
        $customer_data = $this->data_repository->getCustomerData($request['id']);

        return $this->modify_response($customer_data);
    }

    /**
     * @return void
     */
    public function set_up_info(WP_REST_Request $request) {
        $set_up = new SetUp();
        $data = array(
            'default_country' => get_option('woocommerce_default_country', ''),
            'country_edited' => $set_up->getIsCountryEdited(),
            'product_added' => $set_up->getIsProductAdded(),
            'website_edited' => $set_up->getIsWebsiteEdited(),
            'payment_added' => $set_up->getIsPaymentAdded(),
            'countries_list' => Utils::getCountriesStatesList(),
            'payment_mode' => Utils::getTestMode(),
        );

        return $this->modify_response($data);
    }

    /**
     * @return string[]
     */
    public function set_default_country(WP_REST_Request $request) {
        $default_country = $request->get_param('default_country');
        SetUp::updateDefaultCountry($default_country);

        return $this->modify_response();
    }

    public function set_payment_test_mode(WP_REST_Request $request) {
        $mode = $request->get_param('mode');
        Utils::updateTenwebPayengineTestMode($mode);
        $orderActions = new OrderActions();
        $orderActions->change_test_order_statuses();

        return $this->modify_response();
    }

    /**
     * @param $params
     * @param $request
     * @param $key
     *
     * @return bool
     */
    public function validate_unique_positions($params, $request, $key) {
        $params = json_decode($params, true);
        $positions_arr = array();

        foreach ($params as $param) {
            $positions_arr[] = $param['position'];
        }
        $counts = count(array_unique($positions_arr));

        if ($counts !== count($params)) {
            return false;
        }

        return true;
    }

    /**
     * @param $response
     *
     * @return array|mixed
     */
    private function modify_response($response = array()) {
        if ($this->request_from_proxy) {
            return $response;
        }

        return array(
            'msg' => 'success',
            'status' => 200,
            'data' => $response
        );
    }
}
